// IMPORTANT: Keep this path regex and logic in sync with:
//   • URL.isGoogleDocsDocument (Swift)
//   • ContentFetchController.fetchIntegrationOutput (Google Docs guard)
//   • GoogleDocsExport/v1.js
// If you change the pattern here, update those locations as well.
async function __runtimeMainFunction() {
    // 1. Docs‑only pathname:
    //    /document[/u/<account>]/d/<fileId>
    const pathRegex = /^\/document\/(?:u\/(\d+)\/)?d\/([A-Za-z0-9-_]+)/;
    const match = location.pathname.match(pathRegex);
    // If the location isn't a Google Docs document page, exit gracefully.
    if (!match) {
        throw new IntegrationError(
            IntegrationErrorCategory.VALIDATION,
            'Not on Google Docs page',
            { isBenign: true }
        );
    }

    const accountInURL = match[1]; // may be undefined
    const docId = match[2];

    // 2. Determine active account index.
    async function getCurrentAccountIndex() {
        console.debug("[OwnerCheck] Querying ListAccounts for active account...");
        try {
            const resp = await fetch("https://accounts.google.com/ListAccounts?gpsia=1&source=ogb&hl=en", {
                method: "POST",
                credentials: "include",
                headers: { "Content-Type": "application/x-www-form-urlencoded;charset=UTF-8" },
                body: "", // empty body; header + query params satisfy server
            });
            if (!resp.ok) {
                console.warn("[OwnerCheck] ListAccounts HTTP", resp.status);
                return 0;
            }
            const text = await resp.text();
            const parsed = JSON.parse(text);
            const idx = parsed?.[1]?.[0]?.[5] ?? 0;
            const accountCount = parsed?.[1]?.length ?? 0;
            console.debug(`[OwnerCheck] ListAccounts ok (HTTP ${resp.status}) – accounts: ${accountCount}, active index: ${idx}`);
            return idx;
        } catch (err) {
            console.warn("[OwnerCheck] ListAccounts fetch failed", err);
            return 0;
        }
    }

    const accountIndex = accountInURL !== undefined ? Number(accountInURL) : await getCurrentAccountIndex();

    // 3. Build peruserchrome URL using determined index
    const accountSeg = accountIndex ? `/u/${accountIndex}` : "";
    const peruserchromeURL =
        `https://docs.google.com/document${accountSeg}/d/${docId}` +
        `/peruserchrome?id=${docId}&includes_info_params=true&cros_files=false`;

    // 3. Fetch metadata
    const resp = await fetch(peruserchromeURL, {
        method: "GET",
        credentials: "include",
        headers: { "x-same-domain": "1" },
    });

    if (!resp.ok) {
        // If access is forbidden (e.g. viewer-only), we assume the user is *not* the owner.
        if (resp.status === 403) {
            console.warn(`[OwnerCheck] peruserchrome returned HTTP ${resp.status}; assuming not owner.`);
            return { isOwner: false };
        }
        throw new IntegrationError(
            IntegrationErrorCategory.NETWORK,
            'Failed to fetch document metadata',
            { httpStatus: resp.status, url: peruserchromeURL }
        );
    }

    // 4. Strip XSSI prefix and parse
    const raw  = await resp.text();
    const json = JSON.parse(raw.replace(/^\)\]\}'\n?/, ""));

    // 5. Return ownership flag
    return { isOwner: !!json.IS_OWNER };
}
