async function __runtimeMainFunction() {

    // Early-exit guards: only proceed when on a Canvas (instructure) domain *and*
    // we have a clear indication that a PDF could be present (either the URL
    // path contains a known "bucket" segment like /files, /pages, /assignments
    // or the file-preview overlay/modal is open).  Otherwise return an empty
    // fileURL so the caller knows to fall back to HTML scraping.

    // Guard 1 – ensure we're on a Canvas domain.
    const host = window.location.hostname;
    const isCanvasDomain = host === "instructure.com" || host.endsWith(".instructure.com");

    if (!isCanvasDomain) {
        throw new IntegrationError(
            IntegrationErrorCategory.VALIDATION,
            'Not on Canvas domain',
            { isBenign: true }
        );
    }

    // Guard 2 – check path buckets that can host a PDF.
    // IMPORTANT: Keep this implementation in sync with `URL+Canvas.swift` (canHaveCanvasPDF) — any change here likely requires a change there as well.
    const buckets = new Set(["files", "pages", "assignments"]);
    const segments = window.location.pathname.split("/").filter(Boolean);
    const hasBucketSegment = segments.some((seg) => buckets.has(seg));

    // Guard 3 – check if the PDF overlay/modal is currently open.
    const overlayPresent = !!document.querySelector(".ef-file-preview-overlay");

    // If neither a bucket segment nor an overlay is present, there's no point
    // in running the expensive file-ID lookup – exit early.
    if (!hasBucketSegment && !overlayPresent) {
        throw new IntegrationError(
            IntegrationErrorCategory.VALIDATION,
            'Not on a Canvas page with PDF context',
            { isBenign: true }
        );
    }

    async function confirmPDF(url) {
        let headResp;
        try {
            headResp = await fetch(url, { method: "HEAD" });
        } catch (err) {
            throw new IntegrationError(
                IntegrationErrorCategory.NETWORK,
                'Network error on HEAD request: ' + err.message
            );
        }
        if (!headResp.ok) {
            throw new IntegrationError(
                IntegrationErrorCategory.NETWORK,
                'Failed HEAD on public URL',
                { httpStatus: headResp.status, url: url }
            );
        }
        const contentType = headResp.headers.get("content-type") || "";
        if (!contentType.includes("application/pdf")) {
            throw new IntegrationError(
                IntegrationErrorCategory.VALIDATION,
                'File is not a PDF',
                { isBenign: true, context: { contentType: contentType } }
            );
        }
    }

    async function getPublicPDFUrl(fileID) {
        const endpoint = `${window.location.origin}/api/v1/files/${fileID}/public_url`;
        let response;
        try {
            response = await fetch(endpoint, { method: "GET", credentials: "include" });
        } catch (err) {
            throw new IntegrationError(
                IntegrationErrorCategory.NETWORK,
                'Network error fetching public URL: ' + err.message
            );
        }
        if (!response.ok) {
            throw new IntegrationError(
                IntegrationErrorCategory.NETWORK,
                'Failed to get public URL',
                { httpStatus: response.status, url: endpoint }
            );
        }
        const { public_url } = await response.json();
        if (!public_url) {
            throw new IntegrationError(
                IntegrationErrorCategory.NOT_FOUND,
                'No public URL returned from API'
            );
        }

        await confirmPDF(public_url);
        return public_url;
    }

    const pathParts = window.location.pathname.split("/");

    // If we’re on /files/:fileID, use that fileID directly
    const idxFiles = pathParts.indexOf("files");
    if (idxFiles >= 0 && idxFiles + 1 < pathParts.length) {
        const fileID = pathParts[idxFiles + 1];
        return { fileURL: await getPublicPDFUrl(fileID) };
    }

    // Handle URLs like /courses/123/files?preview=297612129 where the file ID is
    // supplied as a query parameter instead of a path segment.
    if (idxFiles >= 0) {
        const params = new URLSearchParams(window.location.search);
        const previewParam = params.get("preview");
        if (previewParam && /^\d+$/.test(previewParam)) {
            return { fileURL: await getPublicPDFUrl(previewParam) };
        }
    }

    // Otherwise, look for the overlay
    const overlayEl = document.querySelector(".ef-file-preview-overlay");
    if (!overlayEl) {
        throw new IntegrationError(
            IntegrationErrorCategory.NOT_FOUND,
            'PDF overlay not found'
        );
    }

    // Attempt to parse fileID from a “download” link
    let fileID = null;
    const downloadLink = overlayEl.querySelector("a.ef-file-preview-header-download");
    if (downloadLink) {
        const href = downloadLink.getAttribute("href") || "";
        const match = href.match(/\/files\/(\d+)\//);
        if (match) {
            fileID = match[1];
        }
    }

    // Fallback: parse fileID from the iframe’s src
    if (!fileID) {
        const iframeEl = overlayEl.querySelector("iframe.ef-file-preview-frame");
        if (iframeEl) {
            const src = iframeEl.getAttribute("src") || "";
            const match = src.match(/\/files\/(\d+)\//);
            if (match) {
                fileID = match[1];
            }
        }
    }

    // If no fileID found, throw error
    if (!fileID) {
        throw new IntegrationError(
            IntegrationErrorCategory.PARSING,
            'Could not extract file ID from overlay'
        );
    }

    // Otherwise, retrieve the public PDF URL
    const fileURL = await getPublicPDFUrl(fileID);
    return { fileURL };
}
