// Detects form elements on a page and returns a count with metadata
// Returns a JSON object with form detection results
function detectFormElements() {
    const result = {
        inputCount: 0,
        textareaCount: 0,
        selectCount: 0,
        checkboxCount: 0,
        radioCount: 0,
        totalFillableElements: 0,
        hasPasswordField: false,
        formCount: 0,
        isLikelySignInPage: false
    };

    // Count <form> elements
    result.formCount = document.querySelectorAll('form').length;

    const excludedInputTypes = new Set([
        'hidden', 'submit', 'button', 'image', 'reset', 'file'
    ]);

    // Helper to check if element is truly interactive (visible and enabled)
    const isInteractive = (el) => {
        if (el.disabled || el.readOnly) return false;

        // Fast layout check
        const rect = el.getBoundingClientRect();
        if (rect.width <= 1 || rect.height <= 1) return false;

        // Stricter visibility check (catches opacity: 0 honeypots)
        // checkVisibility is available in Chrome 105+/Safari 17.4+
        if (el.checkVisibility) {
            return el.checkVisibility({
                checkOpacity: true,
                checkVisibilityCSS: true
            });
        }

        // Fallback for older environments
        const style = window.getComputedStyle(el);
        return style.visibility !== 'hidden' && style.display !== 'none' && style.opacity !== '0';
    };

    // Unified query to prevent double-counting (e.g. input[role="textbox"])
    // querySelectorAll returns a unique list of elements in document order
    const candidates = document.querySelectorAll(
        'input, textarea, select, [contenteditable="true"], [role="textbox"], [role="searchbox"]'
    );

    for (const el of candidates) {
        // Skip non-interactive elements
        if (!isInteractive(el)) continue;

        const tag = el.tagName;

        if (tag === 'INPUT') {
            const type = (el.type || 'text').toLowerCase();
            if (excludedInputTypes.has(type)) continue;

            if (type === 'password') {
                result.hasPasswordField = true;
                // Passwords also count as inputs for fillable purposes
                result.inputCount++; 
            } else if (type === 'checkbox') {
                result.checkboxCount++;
            } else if (type === 'radio') {
                result.radioCount++;
            } else {
                result.inputCount++;
            }
        } else if (tag === 'TEXTAREA') {
            result.textareaCount++;
        } else if (tag === 'SELECT') {
            result.selectCount++;
        } else {
            // Handles contenteditable, role="textbox", role="searchbox"
            // Since we've already handled INPUT/TEXTAREA tags above, these are 
            // exclusively non-standard inputs (like divs)
            result.inputCount++;
        }
    }

    // Calculate total fillable elements
    result.totalFillableElements =
        result.inputCount +
        result.textareaCount +
        result.selectCount +
        result.checkboxCount +
        result.radioCount;

    return JSON.stringify(result);
}

detectFormElements();
