async function __runtimeMainFunction() {
    const WAIT_TIMEOUT_MS = 4_000;
    const POLL_INTERVAL_MS = 150;

    function integrationError(category, message, context = {}) {
        return new IntegrationError(category, message, { context });
    }

    function ensureDocumentAvailable() {
        if (typeof document === 'undefined') {
            throw integrationError(
                IntegrationErrorCategory.VALIDATION,
                'Document is unavailable in this context'
            );
        }
    }

    function sleep(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    async function waitForElement(getter, description) {
        const start = Date.now();
        while (Date.now() - start < WAIT_TIMEOUT_MS) {
            try {
                const el = getter();
                if (el) {
                    return el;
                }
            } catch (err) {
                // Ignore transient DOM errors during polling
            }
            await sleep(POLL_INTERVAL_MS);
        }

        throw integrationError(
            IntegrationErrorCategory.NOT_FOUND,
            `Timed out waiting for ${description}`,
            { description }
        );
    }

    ensureDocumentAvailable();

    if (typeof window === 'undefined' || !window.ytInitialPlayerResponse) {
        throw integrationError(
            IntegrationErrorCategory.VALIDATION,
            'Not on YouTube watch page',
            { isBenign: true }
        );
    }

    // Find and click the transcript button in the description area (any descendant of the transcript container).
    const transcriptButton = await waitForElement(
        () => document.querySelector('div.ytd-video-description-transcript-section-renderer button'),
        'transcript open button'
    );

    if (typeof transcriptButton.click !== 'function') {
        throw integrationError(
            IntegrationErrorCategory.VALIDATION,
            'Transcript trigger is not clickable'
        );
    }

    transcriptButton.click();

    // Wait for the transcript renderer to appear and capture its HTML subtree.
    const transcriptRoot = await waitForElement(
        () => document.querySelector('div#content.ytd-transcript-renderer'),
        'transcript renderer content'
    );

    const transcriptHTML = transcriptRoot?.outerHTML || transcriptRoot?.innerHTML || '';
    const languageCode =
        transcriptRoot?.getAttribute('lang') ||
        transcriptRoot?.getAttribute('xml:lang') ||
        null;

    // TODO: Distill `transcriptHTML` to just the transcript text to reduce the size of
    // the string being sent back to the browser.
    return {
        hasTranscript: Boolean(transcriptHTML),
        languageCode: languageCode,
        transcriptHTML: transcriptHTML
    };
}
