/*───────────────────────────────────────────────────────────────────────────────
  YouTube Transcript Extractor – WHY IT EXISTS, HOW IT WORKS, AND EDGE-CASES
────────────────────────────────────────────────────────────────────────────────

OVERVIEW
▪ We need transcripts for any YouTube watch page.
▪ Most videos respond to a single InnerTube call (get_transcript) that already
  contains an `initialSegments` array.
▪ Some videos, however, **only** return a language-selection footer on the first
  call; the real segments are hidden behind a *continuation* token embedded in
  that footer.  Those are typically videos with multiple subtitle tracks or
  A/B-test behaviour.

This script handles **both** shapes in one pass.

───────────────────────────────────────────────────────────────────────────────
EXECUTION FLOW
1.  📦  **Basic sanity checks**
    • Ensure `ytInitialPlayerResponse` exists (confirms we’re on a watch page).
    • Abort early if the page advertises no caption tracks.

2.  🔍  **Choose a “best” local caption track** (`chosenTrack`)
    • Prefer human-made English ➞ auto-generated English ➞ any non-ASR ➞ first.

3.  🛠  **Assemble the InnerTube request**
    • Build a small protobuf payload (“params”) that mirrors the three-dots
      menu’s “Show transcript” action.
    • Include the extra body fields (`languageCode`, `name`) observed in native
      browser requests.  These appear to make certain videos cooperate.

4.  📡  **Fetch #1 – attempt to get the transcript**
    • POST `/youtubei/v1/get_transcript?key=…`.
    • On success, parse the JSON looking for
      `actions[0].updateEngagementPanelAction.content
        .transcriptRenderer.content.transcriptSearchPanelRenderer.body
        .transcriptSegmentListRenderer.initialSegments`

5.  🚦  **Two code paths from here:**
    a) *Segments found* → great, continue to pagination logic.
    b) *Segments missing* → look in
       `footer.transcriptFooterRenderer.languageMenu.sortFilterSubMenuRenderer`
       for `subMenuItems`.
       ▸ Pick the first English track (or default to the first item).
       ▸ Extract its `reloadContinuationData.continuation` string.
       ▸ **Fetch #2** with that continuation to obtain real segments.

6.  🔄  **Pagination (continuations)**
    • Walk through `continuationCommand.token` fields in the last segment of
      each response, re-fetching until there’s no more token.

7.  🧹  **Normalize segments → cues**
    • Map each `transcriptSegmentRenderer` into `{ start, end, text, timestamp }`
      where `start/end` are seconds and `timestamp` is human-readable.

8.  📤  **Return value**
    ```json
    {
      hasTranscript: <bool>,
      languageCode: <string>,
      cues: [ { start, end, text, humanReadableTimestamp }, … ]
    }
    ```

ERROR HANDLING
▪ Network / parse errors and unexpected JSON shapes raise `abort()` with
  `[YT-TRANSCRIPT]`-prefixed messages so the caller can discriminate them.
▪ Pagination loop terminates gracefully on any subsequent HTTP error.

WHY WE DO THE “LANGUAGE MENU” FALLBACK
▪ During manual QA we found ~10-15 % of videos (often those with multiple human
  subtitle tracks) return *only* the language menu on the first call; the mobile
  site triggers the same behaviour.
▪ Re-issuing the request with the correct `continuation` sidesteps this.
──────────────────────────────────────────────────────────────────────────────*/

async function __runtimeMainFunction() {
    // Early-exit if `ytInitialPlayerResponse` is missing.
    // This global has been present on YouTube watch pages since ~2017 and is
    // relied upon by many scraping tools (youtube-dl, yt-dlp, pytube, etc.) to
    // identify a standard video page and access metadata. If it’s absent, we’re
    // almost certainly not on a normal watch page, so further processing would be
    // wasted.
    if (!window.ytInitialPlayerResponse) {
        throw new IntegrationError(
            IntegrationErrorCategory.VALIDATION,
            'Not on YouTube watch page',
            { isBenign: true }
        );
    }

    /* ── small helpers ─────────────────────────────────────────────── */
    const msToSeconds = ms => Math.round(ms) / 1000;

    const formatTimestamp = seconds => {
        const h = Math.floor(seconds / 3600);
        const m = Math.floor((seconds % 3600) / 60);
        const s = Math.floor(seconds % 60);
        return h
            ? `${h}:${m.toString().padStart(2, '0')}:${s.toString().padStart(2, '0')}`
            : `${m}:${s.toString().padStart(2, '0')}`;
    };


    const playerResponse = window.ytInitialPlayerResponse;
    const trackList      = playerResponse.captions?.playerCaptionsTracklistRenderer;

    if (!trackList?.captionTracks?.length) {
        throw new IntegrationError(
            IntegrationErrorCategory.NOT_FOUND,
            'No caption tracks available',
            { isBenign: true }
        );
    }

    const tracks = trackList.captionTracks;

    /* ── choose the best local track ──────────────────────────────── */
    const isEnglish       = t => t.languageCode?.startsWith('en');
    const isAutoGenerated = t => t.kind === 'asr';

    const chosenTrack =
        tracks.find(t =>  isEnglish(t) && !isAutoGenerated(t)) ||
        tracks.find(t =>  isEnglish(t) &&  isAutoGenerated(t)) ||
        tracks.find(t => !isAutoGenerated(t))                  ||
        tracks[0];

    if (!chosenTrack) {
        throw new IntegrationError(
            IntegrationErrorCategory.NOT_FOUND,
            'Could not choose a caption track',
            { context: { source: 'YT-TRANSCRIPT-V2' } }
        );
    }

    /* ── gather YouTube “InnerTube” info ───────────────────────────── */
    const ytConfig = window.ytcfg;
    const apiKey   = ytConfig?.get('INNERTUBE_API_KEY');
    const context  = ytConfig?.get('INNERTUBE_CONTEXT');
    const videoId  = ytConfig?.get('VIDEO_ID') ||
                     new URLSearchParams(location.search).get('v');

    if (!apiKey || !context || !videoId) {
        throw new IntegrationError(
            IntegrationErrorCategory.VALIDATION,
            'INNERTUBE keys missing – YouTube bootstrap incomplete',
            { context: { source: 'YT-TRANSCRIPT-V2', hasApiKey: !!apiKey, hasContext: !!context, hasVideoId: !!videoId } }
        );
    }

    /* minimal proto encoder (varint + strings) */
    const textEncoder  = new TextEncoder();
    const makeTag      = (field, wireType = 2) => (field << 3) | wireType;

    const encodeVarInt = value => {
        const bytes = [];
        while (value > 0x7f) {
            bytes.push((value & 0x7f) | 0x80);
            value >>= 7;
        }
        bytes.push(value);
        return bytes;
    };

    const encodeString = string => {
        const buf = textEncoder.encode(string);
        return [...encodeVarInt(buf.length), ...buf];
    };

    /* ── build "params" for the chosen track ───────────────────────── */
    const langParamsBytes = [
        makeTag(1), ...encodeString(isAutoGenerated(chosenTrack) ? 'asr' : ''),
        makeTag(2), ...encodeString(chosenTrack.languageCode),
        makeTag(3), 0x00 // variant = ''
    ];
    const langParamsBase64 = btoa(String.fromCharCode(...langParamsBytes));

    const paneId      = 'engagement-panel-searchable-transcript-search-panel0';
    const paramsBytes = [
        makeTag(1), ...encodeString(videoId),
        makeTag(2), ...encodeString(langParamsBase64),
        makeTag(3, 0), 1,
        makeTag(5), ...encodeString(paneId),
        makeTag(7, 0), 1,
        makeTag(8, 0), 1
    ];
    const chosenParams = btoa(String.fromCharCode(...paramsBytes));

    /* ── function to fetch transcript JSON ─────────────────────────── */
    async function fetchTranscriptJson(paramsBase64) {
        const url = `https://www.youtube.com/youtubei/v1/get_transcript?key=${apiKey}`;

        let res;
        try {
            res = await fetch(
                url,
                {
                    method: 'POST',
                    credentials: 'same-origin',
                    headers: { 'content-type': 'application/json' },
                    body: JSON.stringify({
                        context,
                        params: paramsBase64,
                        externalVideoId: videoId,
                        // Extra fields (matching "Request #1") to help some videos
                        languageCode: chosenTrack.languageCode,
                        name: 'CC (English)'
                    })
                }
            );
        } catch (err) {
            // Network error before any HTTP response (e.g. offline, CORS block).
            const online = typeof navigator !== 'undefined' ? navigator.onLine : 'unknown';
            throw new IntegrationError(
                IntegrationErrorCategory.NETWORK,
                'Fetch failed: ' + (err?.message ?? err),
                { context: { onLine: online, source: 'YT-TRANSCRIPT-V2' } }
            );
        }

        if (!res.ok) {
            const statusText = res.statusText ? ` ${res.statusText}` : "";

            throw new IntegrationError(
                IntegrationErrorCategory.NETWORK,
                'InnerTube request failed',
                { httpStatus: res.status, url: url, context: { source: 'YT-TRANSCRIPT-V2' } }
            );
        }
        return res.json();
    }

    /* ── main fetch attempt with chosen track ─────────────────────── */
    let responseJson;
    try {
        responseJson = await fetchTranscriptJson(chosenParams);
    } catch (error) {
        throw new IntegrationError(
            IntegrationErrorCategory.PARSING,
            'Failed to parse InnerTube response: ' + error.message,
            { context: { source: 'YT-TRANSCRIPT-V2' } }
        );
    }

    /* ── parse out segments from JSON ─────────────────────────────── */
    function getInitialSegments(json) {
        return json.actions?.[0]?.updateEngagementPanelAction?.content
            ?.transcriptRenderer?.content
            ?.transcriptSearchPanelRenderer?.body
            ?.transcriptSegmentListRenderer?.initialSegments;
    }

    let segments = getInitialSegments(responseJson);

    // If no segments found, see if there's a “language menu” to re-select track
    if (!segments?.length) {
        // Look in the transcript footer for subMenuItems
        const subMenuItems = responseJson.actions?.[0]
            ?.updateEngagementPanelAction?.content
            ?.transcriptRenderer?.content
            ?.transcriptSearchPanelRenderer?.footer
            ?.transcriptFooterRenderer?.languageMenu
            ?.sortFilterSubMenuRenderer?.subMenuItems;

        if (Array.isArray(subMenuItems) && subMenuItems.length > 0) {
            // Try to find an English item in the subMenu
            const engItem = subMenuItems.find(item =>
                item.title?.toLowerCase().includes('english')
            ) || subMenuItems[0]; // fallback to first if no explicit English

            const continuationStr =
                engItem?.continuation?.reloadContinuationData?.continuation;
            if (continuationStr) {
                // fetch again with that continuation
                try {
                    responseJson = await fetchTranscriptJson(continuationStr);
                    segments = getInitialSegments(responseJson);
                } catch (err) {
                    // If that fails, just log it and continue
                    console.warn('[YT-TRANSCRIPT] Language-menu fetch failed:', err);
                }
            }
        }
    }

    // If we still have no segments, bail out
    if (!segments?.length) {
        throw new IntegrationError(
            IntegrationErrorCategory.NOT_FOUND,
            'No transcript segments found'
        );
    }

    let allSegments = [...segments];

    /* ── loop to fetch any "continuation" pages ───────────────────── */
    let token =
        segments.at(-1)?.continuation?.continuationCommand?.token ||
        segments.at(-1)?.transcriptContinuationRenderer
               ?.continuationEndpoint?.continuationCommand?.token;

    while (token) {
        let page;
        try {
            const contParams = btoa(String.fromCharCode(...[
                makeTag(3), ...encodeString(token)
            ]));
            page = await fetchTranscriptJson(contParams);
        } catch {
            break; // gracefully stop on any error
        }

        const items = page.onResponseReceivedActions?.[0]
            ?.appendContinuationItemsAction?.continuationItems ?? [];

        allSegments.push(...items);

        token =
            items.at(-1)?.continuation?.continuationCommand?.token ||
            items.at(-1)?.transcriptContinuationRenderer
                 ?.continuationEndpoint?.continuationCommand?.token;
    }

    /* ── finalize the cues ─────────────────────────────────────────── */
    const cues = allSegments
        .filter(s => s.transcriptSegmentRenderer)
        .map(s => {
            const r     = s.transcriptSegmentRenderer;
            const text  = r.snippet?.runs?.map(x => x.text).join('') ?? '';
            const start = msToSeconds(+r.startMs);
            const end   = msToSeconds(+r.endMs);

            return {
                start,
                end,
                text,
                humanReadableTimestamp: formatTimestamp(start)
            };
        });

    if (!cues.length) {
        throw new IntegrationError(
            IntegrationErrorCategory.NOT_FOUND,
            'No transcript cues generated'
        );
    }

    /* ── return the result ─────────────────────────────────────────── */
    return {
        hasTranscript: true,
        languageCode: chosenTrack.languageCode,
        cues
    };
}
