async function __runtimeMainFunction() {
    // Early-exit if `ytInitialPlayerResponse` is missing.
    // This global has been present on YouTube watch pages since ~2017 and is
    // relied upon by many scraping tools (youtube-dl, yt-dlp, pytube, etc.) to
    // identify a standard video page and access metadata. If it’s absent, we’re
    // almost certainly not on a normal watch page, so further processing would be
    // wasted.
    if (typeof ytInitialPlayerResponse === 'undefined' || !ytInitialPlayerResponse) {
        throw new IntegrationError(
            IntegrationErrorCategory.VALIDATION,
            'Not on YouTube watch page',
            { isBenign: true }
        );
    }

    const videoId = ytInitialPlayerResponse?.videoDetails?.videoId ||
        new URLSearchParams(location.search).get('v');

    const apiKey = typeof ytcfg !== 'undefined' && ytcfg.get ? ytcfg.get('INNERTUBE_API_KEY') : null;
    if (!videoId || !apiKey) {
        throw new IntegrationError(
            IntegrationErrorCategory.VALIDATION,
            'Missing video id or API key',
            { isBenign: true }
        );
    }

    let playerResp;
    try {
        playerResp = await fetch(`https://www.youtube.com/youtubei/v1/player?key=${apiKey}`, {
            method: 'POST',
            credentials: 'same-origin',
            headers: { 'content-type': 'application/json' },
            body: JSON.stringify({
                context: { client: { clientName: 'ANDROID', clientVersion: '20.10.38' } },
                videoId,
            }),
        });
    } catch (networkErr) {
        throw new IntegrationError(
            IntegrationErrorCategory.NETWORK,
            'Transcript fetch failed: ' + networkErr.message
        );
    }

    if (!playerResp.ok) {
        throw new IntegrationError(
            IntegrationErrorCategory.NETWORK,
            'Player API failed',
            { httpStatus: playerResp.status }
        );
    }

    const playerJson = await playerResp.json();
    const transcriptTracks = playerJson?.captions?.playerCaptionsTracklistRenderer?.captionTracks;
    if (!Array.isArray(transcriptTracks) || transcriptTracks.length === 0) {
        throw new IntegrationError(
            IntegrationErrorCategory.NOT_FOUND,
            'No caption tracks found',
            { isBenign: true }
        );
    }

    let chosenTrack = transcriptTracks.find(
        t => t.languageCode && t.languageCode.startsWith('en')
    );

    if (!chosenTrack) {
        chosenTrack = transcriptTracks[0];
    }
    if (!chosenTrack?.baseUrl) {
        throw new IntegrationError(
            IntegrationErrorCategory.NOT_FOUND,
            'No caption track URL',
            { isBenign: true }
        );
    }

    let response;
    try {
        response = await fetch(chosenTrack.baseUrl, { credentials: 'same-origin' });
    } catch (networkErr) {
        throw new IntegrationError(
            IntegrationErrorCategory.NETWORK,
            'Transcript fetch failed: ' + networkErr.message
        );
    }
    if (!response.ok) {
        throw new IntegrationError(
            IntegrationErrorCategory.NETWORK,
            'Transcript fetch failed',
            { httpStatus: response.status, url: chosenTrack.baseUrl }
        );
    }

    const transcriptXML = await response.text();

    if (!transcriptXML.trim()) {
        throw new IntegrationError(
            IntegrationErrorCategory.NOT_FOUND,
            'EMPTY_TRANSCRIPT',
            { context: { source: 'YT-TRANSCRIPT-V1' } }
        );
    }

    return { transcriptXML };
}
