function fillForm(validateOnly, allowedTagsJson, allowedInputTypesJson, allowedRolesJson, encodedCommands) {
    const config = {
        validateOnly: validateOnly,
        allowedTags: JSON.parse(allowedTagsJson),
        allowedInputTypes: JSON.parse(allowedInputTypesJson),
        allowedRoles: JSON.parse(allowedRolesJson)
    };

    function createResult(success, errorType, message, fieldName, fieldLabel, elementType) {
        return {
            success: success,
            errorType: errorType || null,
            message: message,
            fieldName: fieldName,
            fieldLabel: fieldLabel,
            elementType: elementType || null
        };
    }

    function findElement(xpath) {
        try {
            const result = Document.prototype.evaluate.call(document, xpath, document, null, XPathResult.UNORDERED_NODE_SNAPSHOT_TYPE, null);
            const elementCount = result.snapshotLength;
            if (elementCount === 1) return result.snapshotItem(0);
            if (elementCount > 1) throw new Error('multiple_elements_found');
            return null;
        } catch (error) {
            throw new Error('invalid_selector');
        }
    }

    // extract the label from the element, if it exists.
    // empty string if
    function extractLabel(element) {
        const enclosingLabel = element.closest('label');
        if (enclosingLabel) {
            return enclosingLabel.textContent?.trim() || '';
        }
        if (element.id) {
            const labelFor = Document.prototype.querySelector.call(document, `label[for="${CSS.escape(element.id)}"]`);
            if (labelFor) {
                return labelFor.textContent?.trim() || '';
            }
        }
        const ariaLabel = element.getAttribute('aria-label');
        if (ariaLabel) {
            return ariaLabel.trim();
        }
        return null;
    }
    
    // checkboxes use their label as the name
    // radios use their radio group label as the name
    // returns null if the element has no label
    function extractFieldNameLabel(element) {
        const type = (element.getAttribute('type') || 'text').toLowerCase();
        const role = (element.getAttribute('role') || '').toLowerCase();
        
        const isRadio = element.tagName === 'INPUT' && type === 'radio';
        const isRadioRole = ['DIV', 'SPAN', 'P'].includes(element.tagName) && role === 'radio';
        if (isRadio || isRadioRole) {
            // use the label for the radiogroup (its one of the ancestors)
            const radioGroup = element.closest('[role="radiogroup"]');
            if (radioGroup) {
                return extractLabel(radioGroup);
            }
            
            return null;
        }
        
        const isCheckbox = element.tagName === 'INPUT' && type === 'checkbox';
        const isCheckboxRole = ['DIV', 'SPAN', 'P'].includes(element.tagName) && role === 'checkbox';
        
        if (isCheckbox || isCheckboxRole) {
            return extractLabel(element);
        }

        return null;
    }

    // This is the value of the field; appears in the proposal view.
    function extractFieldValueLabel(element, elementValue) {
        // for <select>, the label is the option's text content
        if (element.tagName === 'SELECT') {
            const opt = element.querySelector(`option[value="${CSS.escape(elementValue)}"]`);
            return opt ? (opt.label || opt.textContent) : elementValue;
        }

        const type = (element.getAttribute('type') || 'text').toLowerCase();
        const role = (element.getAttribute('role') || '').toLowerCase();

        // checkboxes always get a 'true' or 'false'
        // radios get their actual label
        const isRadioCheckbox = element.tagName === 'INPUT' && ['radio', 'checkbox'].includes(type);
        const isRadioCheckboxRole = ['DIV', 'SPAN', 'P'].includes(element.tagName) && ['radio', 'checkbox'].includes(role);
        if (isRadioCheckbox || isRadioCheckboxRole) {
            const shouldBeChecked = ['true', '1', 'on'].includes(String(elementValue).toLowerCase());

            const isRadio = type === 'radio' || role === 'radio';
            if (isRadio) {
                return extractLabel(element) || 'true';
            }

            return shouldBeChecked ? 'true' : 'false';
        }

        // For contenteditable and other role-based elements, return the value as-is
        if (['DIV', 'SPAN', 'P'].includes(element.tagName) && (element.hasAttribute('contenteditable') || element.hasAttribute('role'))) {
            return elementValue.trim();
        }

        return elementValue;
    }
    
    function extractElementType(element) {
        if (element.tagName === 'INPUT') {
            return (element.getAttribute('type') || 'text').toLowerCase();
        } else if (['DIV', 'SPAN', 'P'].includes(element.tagName) && element.hasAttribute('contenteditable')) {
            return 'contenteditable';
        } else if (['DIV', 'SPAN', 'P'].includes(element.tagName) && element.hasAttribute('role')) {
            return element.getAttribute('role').toLowerCase();
        }

        return element.tagName.toLowerCase();
    }

    function processCommand({ id: xpath, value: elementValue, fieldName }) {
        try {
            const element = findElement(xpath);
            if (!element) {
                return createResult(false, 'element_not_found', `Element with xpath "${xpath}" not found.`, fieldName, "", null);
            }

            // Extract element type (input type or tag name as fallback)
            let elementType = extractElementType(element)
            fieldName = extractFieldNameLabel(element) || fieldName;
            const fieldLabel = extractFieldValueLabel(element, elementValue)

            if (config.validateOnly) {
                const rect = element.getBoundingClientRect();
                if (rect.width === 0 || rect.height === 0) {
                    return createResult(false, 'element_not_visible', `Element "${xpath}" is not visible`, fieldName, fieldLabel, elementType);
                }

                // Check if element is allowed (traditional form elements or DIV/SPAN/P elements)
                const isAllowedElement = config.allowedTags.includes(element.tagName) ||
                    (['DIV', 'SPAN', 'P'].includes(element.tagName) && (element.hasAttribute('contenteditable') || element.hasAttribute('role')));

                if (!isAllowedElement) {
                    return createResult(false, 'not_form_input_element', `Element "${xpath}" is not a form input element`, fieldName, fieldLabel, elementType);
                }

                const type = (element.getAttribute('type') || 'text').toLowerCase();
                if (element.tagName === 'INPUT' && !config.allowedInputTypes.includes(type) && !['radio', 'checkbox'].includes(type)) {
                    return createResult(false, 'invalid_input_type', `Not a valid input type: ${type}`, fieldName, fieldLabel, elementType);
                }

                // Validate role attribute for role-based elements
                if (['DIV', 'SPAN', 'P'].includes(element.tagName) && element.hasAttribute('role')) {
                    const role = (element.getAttribute('role') || '').toLowerCase();
                    if (!config.allowedRoles.includes(role)) {
                        return createResult(false, 'invalid_role_type', `Not a valid role: ${role}`, fieldName, fieldLabel, elementType);
                    }
                }

                return createResult(true, null, `Valid element: ${fieldName}`, fieldName, fieldLabel, elementType);
            }

            const type = (element.getAttribute('type') || 'text').toLowerCase();
            const role = (element.getAttribute('role') || '').toLowerCase();
            if (element.tagName === 'INPUT' && ['radio', 'checkbox'].includes(type)) {
                const shouldBeChecked = ['true', '1', 'on'].includes(String(elementValue).toLowerCase());
                if (shouldBeChecked !== element.checked) element.click();
                const action = shouldBeChecked ? 'checked' : 'unchecked';
                return createResult(true, null, `Successfully ${action} ${xpath}`, fieldName, fieldLabel, elementType);
            } else if (['DIV', 'SPAN', 'P'].includes(element.tagName) && ['radio', 'checkbox'].includes(role)) {
                // Use click for checkbox/radio roles
                const shouldBeChecked = ['true', '1', 'on'].includes(String(elementValue).toLowerCase());
                const isCurrentlyChecked = element.getAttribute('aria-checked') === 'true' || element.classList.contains('checked');
                if (shouldBeChecked !== isCurrentlyChecked) element.click();
                const action = shouldBeChecked ? 'checked' : 'unchecked';
                return createResult(true, null, `Successfully ${action} ${xpath}`, fieldName, fieldLabel, elementType);
            } else if (['DIV', 'SPAN', 'P'].includes(element.tagName) && (element.hasAttribute('contenteditable') || element.hasAttribute('role'))) {
                // Use textContent for contenteditable and non-checkbox/radio role elements to prevent code injection
                element.textContent = elementValue;
                element.dispatchEvent(new Event('input', { bubbles: true }));
                element.dispatchEvent(new Event('change', { bubbles: true }));
                return createResult(true, null, `Applied to ${xpath}`, fieldName, fieldLabel, elementType);
            } else {
                element.value = elementValue;
                element.dispatchEvent(new Event('input', { bubbles: true }));
                element.dispatchEvent(new Event('change', { bubbles: true }));
                return createResult(true, null, `Applied to ${xpath}`, fieldName, fieldLabel, elementType);
            }
        } catch (error) {
            const errorType = error.message === 'multiple_elements_found' ? 'multiple_elements_found' :
                             error.message === 'invalid_selector' ? 'invalid_selector' : 'js_execution_error';
            const action = config.validateOnly ? 'validating' : 'applying to';
            return createResult(false, errorType, `Error ${action} element "${xpath}": ${error.message}`, fieldName, "", null);
        }
    }
    
    function base64ToUtf8(base64) {
      const bytes = Uint8Array.fromBase64(base64);
      const decoder = new TextDecoder('utf-8');
      return decoder.decode(bytes);
    }

    try {
        const commands = JSON.parse(base64ToUtf8(encodedCommands));
        const results = commands.map(processCommand);
        return results;
    } catch (error) {
        const errorResults = [createResult(false,
                                           'js_execution_error',
                                           `Error processing form commands: ${error.message}`,
                                           'unknown',
                                           '',
                                           null)];
        return errorResults;
    }
}
